import gradio as gr
from datetime import datetime, timedelta

def create_job_browser_ui(fetch_jobs):
    """
    Initialize and return the Gradio interface for job browsing.
    
    Args:
        fetch_jobs: Function that fetches and displays jobs based on date range
    Returns:
        gr.Blocks: The configured Gradio interface
    """
    
    with gr.Blocks(title="EOSCONNECT Job Browser") as demo:
        gr.Markdown("# 🔍 EOSCONNECT Job Browser")
        gr.Markdown("Filter and view build jobs by date range")
        
        with gr.Row():
            from_date_input = gr.Textbox(
                label="From Date (YYYY-MM-DD)",
                placeholder="2024-01-01",
                value=(datetime.now() - timedelta(days=30)).strftime("%Y-%m-%d")
            )
            to_date_input = gr.Textbox(
                label="To Date (YYYY-MM-DD)",
                placeholder="2024-12-31",
                value=datetime.now().strftime("%Y-%m-%d")
            )
        
        fetch_button = gr.Button("🔄 Fetch Jobs", variant="primary")
        
        jobs_table = gr.Dataframe(
            headers=['id', 'result', 'timeStarted', 'timeEnded', 'task', 'material'],
            label="Jobs",
            interactive=False,
            wrap=True
        )
        
        # Connect button to function
        fetch_button.click(
            fn=fetch_jobs,
            inputs=[from_date_input, to_date_input],
            outputs=jobs_table
        )
        
        # Load initial data
        demo.load(
            fn=fetch_jobs,
            inputs=[from_date_input, to_date_input],
            outputs=jobs_table
        )
    
    return demo




def create_job_status_ui(fetch_job_status, clear_last_job_id_tracker):
    """
    Initialize and return the Gradio interface for job status checking.
    
    Args:
        fetch_job_status: Function that fetches and displays job status
        clear_last_job_id_tracker: Function that clears the last job ID tracker
    Returns:
        gr.Blocks: The configured Gradio interface
    """
    with gr.Blocks(title="EOSCONNECT Last Job Status") as job_status_demo:
        gr.Markdown("# 🔍 EOSCONNECT Last Job Status Checker")
        gr.Markdown("Check the status of the most recent job and view any user messages")
        
        check_button = gr.Button("🔄 Check Last Job Status", variant="primary", size="lg")
        clear_button = gr.Button("Clear the last job ID", variant="secondary", size="lg")
        
        with gr.Row():
            with gr.Column():
                gr.Markdown("### Job Information")
                job_info_table = gr.Dataframe(
                    headers=['Field', 'Value'],
                    label="Job Details",
                    interactive=False,
                    wrap=True
                )
            
            with gr.Column():
                gr.Markdown("### User Messages")
                user_messages_table = gr.Dataframe(
                    headers=['Severity', 'timeRaised', 'message', 'additionalInfo'],
                    label="Messages",
                    interactive=False,
                    wrap=True
                )
        
        # Connect button to function
        check_button.click(
            fn=fetch_job_status,
            inputs=None,
            outputs=[job_info_table, user_messages_table]
        )
        
        clear_button.click(
            fn=clear_last_job_id_tracker,
            inputs=None,
            outputs=[job_info_table, user_messages_table]
        )
        
        # Load initial data
        job_status_demo.load(
            fn=fetch_job_status,
            inputs=None,
            outputs=[job_info_table, user_messages_table]
        )
    
    return job_status_demo